﻿using System;
using System.Collections.Generic;
using System.Threading;
using FyndSharp.Utilities.Threading;
using gov.va.med.VBECS.Communication.Common;
using gov.va.med.VBECS.Communication.Protocols;

namespace gov.va.med.VBECS.Communication.Channels
{
    /// <summary>
    /// This class adds SendMessageAndWaitForResponse(...) and SendAndReceiveMessage methods
    /// to a IMessenger for synchronous request/response style messaging.
    /// It also adds queued processing of incoming messages.
    /// </summary>
    /// <typeparam name="T">Type of IMessenger object to use as underlying communication</typeparam>
    public class RequestReplyMessager<T> : IMessager, IDisposable where T : IMessager
    {
        #region Public events

        /// <summary>
        /// This event is raised when a new message is received from underlying messenger.
        /// </summary>
        public event EventHandler<MessageEventArgs> MessageReceived;

        /// <summary>
        /// This event is raised when a new message is sent without any error.
        /// It does not guaranties that message is properly handled and processed by remote application.
        /// </summary>
        public event EventHandler<MessageEventArgs> MessageSent;

        #endregion

        #region Public properties

        /// <summary>
        /// Gets the underlying IMessenger object.
        /// </summary>
        public T Messager { get; private set; }

        /// <summary>
        /// Timeout value as milliseconds to wait for a receiving message on 
        /// SendMessageAndWaitForResponse and SendAndReceiveMessage methods.
        /// Default value: 60000 (1 minute).
        /// </summary>
        public int Timeout { get; set; }

        /// <summary>
        /// Gets/sets wire protocol that is used while reading and writing messages.
        /// </summary>
        public IProtocol Protocol
        {
            get { return Messager.Protocol; }
            set { Messager.Protocol = value; }
        }

        /// <summary>
        /// Gets the time of the last successfully received message.
        /// </summary>
        public DateTime LastReceivedTime
        {
            get { return Messager.LastReceivedTime; }
        }

        /// <summary>
        /// Gets the time of the last successfully received message.
        /// </summary>
        public DateTime LastSentTime
        {
            get { return Messager.LastSentTime; }
        }

        #endregion

        #region Private fields

        /// <summary>
        /// Default Timeout value.
        /// </summary>
        private const int DEFAULT_TIMEOUT = 60000;

        /// <summary>
        /// This object is used to process incoming messages sequentially.
        /// </summary>
        private readonly SequentialItemProcessor<IMessage> _incomingMessageProcessor;

        /// <summary>
        /// This object is used for thread synchronization.
        /// </summary>
        private readonly Object _syncObject = new Object();

        /// <summary>
        /// This messages are waiting for a response those are used when 
        /// SendMessageAndWaitForResponse is called.
        /// Key: MessageID of waiting request message.
        /// Value: A WaitingMessage instance.
        /// </summary>
        private readonly SortedList<string, WaitingMessage> _waitingMessages;

        #endregion

        #region Constructor

        /// <summary>
        /// Creates a new RequestReplyMessenger.
        /// </summary>
        /// <param name="theMessager">IMessenger object to use as underlying communication</param>
        public RequestReplyMessager(T theMessager)
        {
            Messager = theMessager;
            theMessager.MessageReceived += messager_message_received;
            theMessager.MessageSent += messager_message_sent;
            _incomingMessageProcessor = new SequentialItemProcessor<IMessage>(FireMessageReceivedEvent);
            _waitingMessages = new SortedList<string, WaitingMessage>();
            Timeout = DEFAULT_TIMEOUT;
        }

        #endregion

        #region Public methods

        #region IDisposable Members

        /// <summary>
        /// Calls Stop method of this object.
        /// </summary>
        public void Dispose()
        {
            Stop();
        }

        #endregion

        #region IMessager Members

        /// <summary>
        /// Sends a message.
        /// </summary>
        /// <param name="aMessage">Message to be sent</param>
        public void Send(IMessage aMessage)
        {
            Messager.Send(aMessage);
        }

        #endregion

        /// <summary>
        /// Starts the messenger.
        /// </summary>
        public virtual void Start()
        {
            _incomingMessageProcessor.Start();
        }

        /// <summary>
        /// Stops the messenger.
        /// Cancels all waiting threads in SendMessageAndWaitForResponse method and stops message queue.
        /// SendMessageAndWaitForResponse method throws exception if there is a thread that is waiting for response message.
        /// Also stops incoming message processing and deletes all messages in incoming message queue.
        /// </summary>
        public virtual void Stop()
        {
            _incomingMessageProcessor.Stop();

            //Pulse waiting threads for incoming messages, since underlying messenger is disconnected
            //and can not receive messages anymore.
            lock (_syncObject)
            {
                foreach (var waitingMessage in _waitingMessages.Values)
                {
                    waitingMessage.Status = WaitingMessageStatus.Cancelled;
                    waitingMessage.WaitEvent.Set();
                }

                _waitingMessages.Clear();
            }
        }

        /// <summary>
        /// Sends a message and waits a response for that message.
        /// </summary>
        /// <remarks>
        /// Response message is matched with RepliedMessageId property, so if
        /// any other message (that is not reply for sent message) is received
        /// from remote application, it is not considered as a reply and is not
        /// returned as return value of this method.
        /// 
        /// MessageReceived event is not raised for response messages.
        /// </remarks>
        /// <param name="aMessage">message to send</param>
        /// <returns>Response message</returns>
        public IMessage SendMessageAndWaitForResponse(IMessage aMessage)
        {
            return SendMessageAndWaitForResponse(aMessage, Timeout);
        }

        /// <summary>
        /// Sends a message and waits a response for that message.
        /// </summary>
        /// <remarks>
        /// Response message is matched with RepliedMessageId property, so if
        /// any other message (that is not reply for sent message) is received
        /// from remote application, it is not considered as a reply and is not
        /// returned as return value of this method.
        /// 
        /// MessageReceived event is not raised for response messages.
        /// </remarks>
        /// <param name="aMessage">message to send</param>
        /// <param name="theTimeoutMilliseconds">Timeout duration as milliseconds.</param>
        /// <returns>Response message</returns>
        /// <exception cref="TimeoutException">Throws TimeoutException if can not receive reply message in timeout value</exception>
        /// <exception cref="CommunicationException">Throws CommunicationException if communication fails before reply message.</exception>
        public IMessage SendMessageAndWaitForResponse(IMessage aMessage, int theTimeoutMilliseconds)
        {
            //Create a waiting message record and add to list
            var waitingMessage = new WaitingMessage();
            lock (_syncObject)
            {
                _waitingMessages[aMessage.Id] = waitingMessage;
            }

            try
            {
                //Send message
                Messager.Send(aMessage);

                //Wait for response
                waitingMessage.WaitEvent.WaitOne(theTimeoutMilliseconds);

                //Check for exceptions
                switch (waitingMessage.Status)
                {
                    case WaitingMessageStatus.WaitingForResponse:
                        throw new TimeoutException("Timeout occured. Can not received response.");
                    case WaitingMessageStatus.Cancelled:
                        throw new CommunicationException("Disconnected before response received.");
                }

                //return response message
                return waitingMessage.ResponseMessage;
            }
            finally
            {
                //Remove message from waiting messages
                lock (_syncObject)
                {
                    if (_waitingMessages.ContainsKey(aMessage.Id))
                    {
                        _waitingMessages.Remove(aMessage.Id);
                    }
                }
            }
        }

        #endregion

        #region Private methods

        /// <summary>
        /// Handles MessageReceived event of Messenger object.
        /// </summary>
        /// <param name="sender">Source of event</param>
        /// <param name="e">Event arguments</param>
        private void messager_message_received(object sender, MessageEventArgs e)
        {
            //Check if there is a waiting thread for this message in SendMessageAndWaitForResponse method
            if (!string.IsNullOrEmpty(e.Message.RepliedId))
            {
                WaitingMessage waitingMessage = null;
                lock (_syncObject)
                {
                    if (_waitingMessages.ContainsKey(e.Message.RepliedId))
                    {
                        waitingMessage = _waitingMessages[e.Message.RepliedId];
                    }
                }

                //If there is a thread waiting for this response message, pulse it
                if (waitingMessage != null)
                {
                    waitingMessage.ResponseMessage = e.Message;
                    waitingMessage.Status = WaitingMessageStatus.ResponseReceived;
                    waitingMessage.WaitEvent.Set();
                    return;
                }
            }

            _incomingMessageProcessor.EnqueueMessage(e.Message);
        }

        /// <summary>
        /// Handles MessageSent event of Messenger object.
        /// </summary>
        /// <param name="sender">Source of event</param>
        /// <param name="e">Event arguments</param>
        private void messager_message_sent(object sender, MessageEventArgs e)
        {
            FireMessageSentEvent(e.Message);
        }

        #endregion

        #region Event raising methods

        /// <summary>
        /// Raises MessageReceived event.
        /// </summary>
        /// <param name="theMessage">Received message</param>
        protected virtual void FireMessageReceivedEvent(IMessage theMessage)
        {
            var handler = MessageReceived;
            if (handler != null)
            {
                handler(this, new MessageEventArgs(theMessage));
            }
        }

        /// <summary>
        /// Raises MessageSent event.
        /// </summary>
        /// <param name="theMessage">Received message</param>
        protected virtual void FireMessageSentEvent(IMessage theMessage)
        {
            EventHandler<MessageEventArgs> handler = MessageSent;
            if (handler != null)
            {
                handler(this, new MessageEventArgs(theMessage));
            }
        }

        #endregion

        #region WaitingMessage class

        #region Nested type: WaitingMessage

        /// <summary>
        /// This class is used to store messaging context for a request message
        /// until response is received.
        /// </summary>
// ReSharper disable InconsistentNaming
        private sealed class WaitingMessage
// ReSharper restore InconsistentNaming
        {
            /// <summary>
            /// Creates a new WaitingMessage object.
            /// </summary>
            public WaitingMessage()
            {
                WaitEvent = new ManualResetEvent(false);
                Status = WaitingMessageStatus.WaitingForResponse;
            }

            /// <summary>
            /// Response message for request message 
            /// (null if response is not received yet).
            /// </summary>
            public IMessage ResponseMessage { get; set; }

            /// <summary>
            /// ManualResetEvent to block thread until response is received.
            /// </summary>
            public ManualResetEvent WaitEvent { get; private set; }

            /// <summary>
            /// State of the request message.
            /// </summary>
            public WaitingMessageStatus Status { get; set; }
        }

        #endregion

        #region Nested type: WaitingMessageStatus

        /// <summary>
        /// This enum is used to store the state of a waiting message.
        /// </summary>
        private enum WaitingMessageStatus
        {
            /// <summary>
            /// Still waiting for response.
            /// </summary>
            WaitingForResponse,

            /// <summary>
            /// Message sending is cancelled.
            /// </summary>
            Cancelled,

            /// <summary>
            /// Response is properly received.
            /// </summary>
            ResponseReceived
        }

        #endregion

        #endregion
    }
}